/*************************************************************************
 * The contents of this file are subject to the MYRICOM MX AND GM-2      *
 * MAPPING SOFTWARE AND DOCUMENTATION LICENSE (the "License"); User may  *
 * not use this file except in compliance with the License.  The full    *
 * text of the License can found in mapper directory in LICENSE.TXT      *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

/*
  assumes nodes are named x0-x(num_xbars - 1), h0-h(num_hosts - 1).
*/

#include "lx.h"
#include "lx_tokens.h"
#include "lx_map_file.h"

#define LX_MAP_FILE_MAX_LINE 200
#define LX_MAP_FILE_WHITESPACE "\n\t \r"
#define LX_MAP_FILE_COMMENTS "#;"

#define lx_map_file_error(tokens,e){printf ("%d: ", lx_tokens_get_line (tokens));printf e; printf ("\n"); goto error;}

static lx_node_t*lx_map_file_get_node (lx_map_t*m, char*s, int*xbar)
{
  int d;
  lx_node_t*n;
  
  insist (s && xbar);
  insist (*s == 'x' || *s == 'h');
  insist (m->lx);
  
  *xbar = *s == 'x';
  d = atoi (s + 1);

  if (*xbar)
  {
    if (d < 0 || d > LX_MAX_XBARS)
      return 0;
  
    if (!(n = mi_index2node (m->xbar_array [d])))
    {
      n = lx_new_xbar (m, 0, 0, 0, 0, 0);
      insist (n);
      m->xbar_array [d] = mi_node2index (n);
      n->index = d;
      lx_put (&m->lx->staged_xbars, n);
    }
    return n;
  }
  else
  {
    if (d < 0 || d > LX_MAX_HOSTS)
      return 0;

    if (!(n = mi_index2node (m->host_array [d])))
    {
      n = lx_new_host (m, 0, 0, 0, 0, 0, 0);
      insist (n);
      m->host_array [d] = mi_node2index (n);
      n->index = d;
      lx_put (&m->lx->staged_hosts, n);
    }
    return n;
  }
  except: return 0;
}

static lx_node_t*lx_map_file_read_node (lx_map_t*m, void*tokens, int*xbar)
{
  char*s;
  lx_node_t*n;
  int x1;
  
  insist (m && tokens && xbar);

  if (!(s = lx_tokens_next (tokens)))
    return 0;
  
  if (strcmp (s, "s") && strcmp (s, "h"))
    lx_map_file_error (tokens, ("%s is a bad node type", s));
  
  x1 = !strcmp (s, "s");  

  if (!(s = lx_tokens_next (tokens)))
    lx_map_file_error (tokens, ("expected placeholder"));

  if (!(s = lx_tokens_next (tokens)))
    lx_map_file_error (tokens, ("expected node name"));
  
  if (!(n = lx_map_file_get_node (m, s, xbar)))
    lx_map_file_error (tokens, ("bad node name %s", s));
  
  if ((*xbar && !x1) || (!*xbar && x1))
    lx_map_file_error (tokens, ("type / name mismatch"));

  return n;
  error:
  except: return 0;
}

static int lx_map_file_read_links (lx_map_t*m, void*tokens, lx_node_t*n)
{
  lx_node_t*o;
  char*s;
  int i, num_links, iport, oiport, xbar;
  
  insist (m && tokens && n);
  
  if (!(s = lx_tokens_next (tokens)))
    lx_map_file_error (tokens, ("expected link count"));
  
  num_links = atoi (s);
  if (num_links < 0 || num_links > (n->xbar ? LX_XBAR_SIZE : LX_HOST_SIZE))
    lx_map_file_error (tokens, ("%s is a bad link count", s));

  for (i = 0; i < num_links; i++)
  {
    if (!(s = lx_tokens_next (tokens)))
      lx_map_file_error (tokens, ("expected iport"));
  
    iport = atoi (s);
    if (iport < 0 || iport >= (n->xbar ? LX_XBAR_SIZE : LX_HOST_SIZE))
      lx_map_file_error (tokens, ("%s is a bad iport", s));

    if (!(o = lx_map_file_read_node (m, tokens, &xbar)))
      lx_map_file_error (tokens, ("failed to read node"));
  
    if (!(s = lx_tokens_next (tokens)))
      lx_map_file_error (tokens, ("expected oiport"));
  
    oiport = atoi (s);
    if (oiport < 0 || oiport >= (xbar ? LX_XBAR_SIZE : LX_HOST_SIZE))
      lx_map_file_error (tokens, ("%s is a bad oiport", s));

    insist ((o->xbar && xbar) || (!o->xbar && !xbar));
    
    if (!lx_connect (n, iport, o, oiport))
      lx_map_file_error (tokens, ("something wicked"));
  }
  
  return 1;

  error:
  except: return 0;
}

static int lx_map_file_read_host (lx_map_t*m, void*tokens, lx_node_t*n)
{
  char*s;
  int i, host_type;
  int a [6];
  
  insist (m && tokens && n);

  if (!lx_map_file_read_links (m, tokens, n))
    goto error;
  
  if (!(s = lx_tokens_next (tokens)) || strcmp (s, "address"))
    lx_map_file_error (tokens, ("expected address tag"));
  
  if (!(s = lx_tokens_next (tokens)) || 6 != sscanf (s, "%x:%x:%x:%x:%x:%x", &a [0], &a [1], &a [2], &a [3], &a [4], &a [5]))
    lx_map_file_error (tokens, ("expected address"));

  for (i = 0; i < 6; i++)
    lx_host_c (n)->mac_address [i] = a [i];
  
  if (!(s = lx_tokens_next (tokens)) || strcmp (s, "hostType"))
    lx_map_file_error (tokens, ("expected hostType tag"));

  if (!(s = lx_tokens_next (tokens)))
    lx_map_file_error (tokens, ("expected hostType"));
  
  host_type = atoi (s);
  if (host_type < 0 || host_type > LX_NUM_HOST_TYPES)
    lx_map_file_error (tokens, ("%s is a bad host type", s));
  lx_host_c (n)->host_type = host_type;

  lx_put (&m->hosts, lx_remove_from_anywhere (n));
  
  return 1;
  error:
  except: return 0;
}


static int lx_map_file_read_xbar (lx_map_t*m, void*tokens, lx_node_t*n)
{
  char*s;
  
  insist (m && tokens && n);
  
  if (!lx_map_file_read_links (m, tokens, n))
    goto error;

  if (!(s = lx_tokens_next (tokens)) || strcmp (s, "ID"))
    lx_map_file_error (tokens, ("expected xbar ID tag"));
  
  if (!(s = lx_tokens_next (tokens)) || 1 != sscanf (s, "%u", &(lx_xbar_c (n)->id)))
    lx_map_file_error (tokens, ("expected id"));
  
  if (!(s = lx_tokens_next (tokens)) || strcmp (s, "decodeID"))
    lx_map_file_error (tokens, ("expected decodeID tag"));

  if (!(s = lx_tokens_next (tokens)))
    lx_map_file_error (tokens, ("expected decodeID"));

  /*ignore it*/
  
  lx_put (&m->xbars, lx_remove_from_anywhere (n));
  
  return 1;
  error:
  except: return 0;
}

int
lx_map_file_read (FILE*fp, lx_map_t*m, lx_node_t**first)
{
  void*tokens = 0;
  lx_node_t*n;
  int xbar;
  int found_host = 0;
  
  insist (fp && m && first);
  insist (m->num_hosts == 0 && m->num_xbars == 0);
  
  if (!(tokens = lx_tokens_new (fp, LX_MAP_FILE_MAX_LINE, LX_MAP_FILE_WHITESPACE, LX_MAP_FILE_COMMENTS)))
    return 0;
  
  while ((n = lx_map_file_read_node (m, tokens, &xbar)))
  {
    if (xbar)
    {
      if (!lx_map_file_read_xbar (m, tokens, n))
	goto error;
    }
    else
    {
      if (!lx_map_file_read_host (m, tokens, n))
	goto error;

      if (!found_host)
      {
	found_host = 1;
	*first = n;
      }
    }
  }

  lx_tokens_free (tokens);
  return 1;

  error:
  lx_tokens_free (tokens);
  except: return 0;
  
}

/*yes for some reason this function duplicates lx_print_node
  almost line for line.*/

static int lx_map_file_print_node (FILE*fp, lx_node_t*n)
{
  int i, c = 1;
  lx_node_t*nn;
  
  insist (fp && n);
  
  if (n->xbar)
  {
    for (c = 0, i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE; i++)
      c += lx_connected (n, i) ? 1 : 0;
  }
  else
  {
    for (c = 0, i = 0; i < LX_HOST_SIZE; i++)
      c += lx_connected (n, i) ? 1 : 0;
  }
  
  fprintf (fp, "%s - " lx_node_format "\n", n->xbar ? "s" : "h", lx_node_args (n));
  fprintf (fp, "%d\n", c);

  for (i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE; i++)
  {
    if ((nn = lx_get_node (n, i)))
    {
      insist (i >= n->first);
      insist (lx_get_port (n, i) >= nn->first);
      fprintf (fp, "%d %s - " lx_node_format " %d\n", i - n->first, nn->xbar ? "s" : "h", lx_node_args (nn), lx_get_port (n, i) - nn->first);
    }
  }
  
  if (n->xbar)
  {
    unsigned id = lx_xbar_c (n)->id;
    fprintf (fp, "ID %u\n", id);
    fprintf (fp, "decodeID %u:%u\n", id / 16, id % 16);
  }
  else
  {
    fprintf (fp, "address " lx_mac_format "\n", lx_mac_args (lx_host_c (n)->mac_address));
    fprintf (fp, "hostType %d\n", lx_host_c (n)->host_type);

  }
  
  fprintf (fp, "\n");
  
  return 1;  
  except: return 0;
}

int lx_map_file_write (FILE*fp, lx_map_t*m)
{
  int i;
  
  insist (fp && m);
  fprintf (fp, "; map_address " lx_mac_format " map_version %u\n", lx_mac_args (m->map_address), m->map_version);

  for (i = 0; i < m->num_hosts; i++)
    lx_map_file_print_node (fp, mi_index2node (m->host_array [i]));
  for (i = 0; i < m->num_xbars; i++)
    lx_map_file_print_node (fp, mi_index2node (m->xbar_array [i]));
  
  return 1;
  except: return 0;
}
